/*! firebase-admin v10.3.0 */
/*!
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Interface representing base properties of a user-enrolled second factor for a
 * `CreateRequest`.
 */
export interface BaseCreateMultiFactorInfoRequest {
    /**
     * The optional display name for an enrolled second factor.
     */
    displayName?: string;
    /**
     * The type identifier of the second factor. For SMS second factors, this is `phone`.
     */
    factorId: string;
}
/**
 * Interface representing a phone specific user-enrolled second factor for a
 * `CreateRequest`.
 */
export interface CreatePhoneMultiFactorInfoRequest extends BaseCreateMultiFactorInfoRequest {
    /**
     * The phone number associated with a phone second factor.
     */
    phoneNumber: string;
}
/**
 * Type representing the properties of a user-enrolled second factor
 * for a `CreateRequest`.
 */
export declare type CreateMultiFactorInfoRequest = CreatePhoneMultiFactorInfoRequest;
/**
 * Interface representing common properties of a user-enrolled second factor
 * for an `UpdateRequest`.
 */
export interface BaseUpdateMultiFactorInfoRequest {
    /**
     * The ID of the enrolled second factor. This ID is unique to the user. When not provided,
     * a new one is provisioned by the Auth server.
     */
    uid?: string;
    /**
     * The optional display name for an enrolled second factor.
     */
    displayName?: string;
    /**
     * The optional date the second factor was enrolled, formatted as a UTC string.
     */
    enrollmentTime?: string;
    /**
     * The type identifier of the second factor. For SMS second factors, this is `phone`.
     */
    factorId: string;
}
/**
 * Interface representing a phone specific user-enrolled second factor
 * for an `UpdateRequest`.
 */
export interface UpdatePhoneMultiFactorInfoRequest extends BaseUpdateMultiFactorInfoRequest {
    /**
     * The phone number associated with a phone second factor.
     */
    phoneNumber: string;
}
/**
 * Type representing the properties of a user-enrolled second factor
 * for an `UpdateRequest`.
 */
export declare type UpdateMultiFactorInfoRequest = UpdatePhoneMultiFactorInfoRequest;
/**
 * The multi-factor related user settings for create operations.
 */
export interface MultiFactorCreateSettings {
    /**
     * The created user's list of enrolled second factors.
     */
    enrolledFactors: CreateMultiFactorInfoRequest[];
}
/**
 * The multi-factor related user settings for update operations.
 */
export interface MultiFactorUpdateSettings {
    /**
     * The updated list of enrolled second factors. The provided list overwrites the user's
     * existing list of second factors.
     * When null is passed, all of the user's existing second factors are removed.
     */
    enrolledFactors: UpdateMultiFactorInfoRequest[] | null;
}
/**
 * Interface representing the properties to update on the provided user.
 */
export interface UpdateRequest {
    /**
     * Whether or not the user is disabled: `true` for disabled;
     * `false` for enabled.
     */
    disabled?: boolean;
    /**
     * The user's display name.
     */
    displayName?: string | null;
    /**
     * The user's primary email.
     */
    email?: string;
    /**
     * Whether or not the user's primary email is verified.
     */
    emailVerified?: boolean;
    /**
     * The user's unhashed password.
     */
    password?: string;
    /**
     * The user's primary phone number.
     */
    phoneNumber?: string | null;
    /**
     * The user's photo URL.
     */
    photoURL?: string | null;
    /**
     * The user's updated multi-factor related properties.
     */
    multiFactor?: MultiFactorUpdateSettings;
    /**
     * Links this user to the specified provider.
     *
     * Linking a provider to an existing user account does not invalidate the
     * refresh token of that account. In other words, the existing account
     * would continue to be able to access resources, despite not having used
     * the newly linked provider to log in. If you wish to force the user to
     * authenticate with this new provider, you need to (a) revoke their
     * refresh token (see
     * https://firebase.google.com/docs/auth/admin/manage-sessions#revoke_refresh_tokens),
     * and (b) ensure no other authentication methods are present on this
     * account.
     */
    providerToLink?: UserProvider;
    /**
     * Unlinks this user from the specified providers.
     */
    providersToUnlink?: string[];
}
/**
 * Represents a user identity provider that can be associated with a Firebase user.
 */
export interface UserProvider {
    /**
     * The user identifier for the linked provider.
     */
    uid?: string;
    /**
     * The display name for the linked provider.
     */
    displayName?: string;
    /**
     * The email for the linked provider.
     */
    email?: string;
    /**
     * The phone number for the linked provider.
     */
    phoneNumber?: string;
    /**
     * The photo URL for the linked provider.
     */
    photoURL?: string;
    /**
     * The linked provider ID (for example, "google.com" for the Google provider).
     */
    providerId?: string;
}
/**
 * Interface representing the properties to set on a new user record to be
 * created.
 */
export interface CreateRequest extends UpdateRequest {
    /**
     * The user's `uid`.
     */
    uid?: string;
    /**
     * The user's multi-factor related properties.
     */
    multiFactor?: MultiFactorCreateSettings;
}
/**
 * The response interface for listing provider configs. This is only available
 * when listing all identity providers' configurations via
 * {@link BaseAuth.listProviderConfigs}.
 */
export interface ListProviderConfigResults {
    /**
     * The list of providers for the specified type in the current page.
     */
    providerConfigs: AuthProviderConfig[];
    /**
     * The next page token, if available.
     */
    pageToken?: string;
}
/**
 * The filter interface used for listing provider configurations. This is used
 * when specifying how to list configured identity providers via
 * {@link BaseAuth.listProviderConfigs}.
 */
export interface AuthProviderConfigFilter {
    /**
     * The Auth provider configuration filter. This can be either `saml` or `oidc`.
     * The former is used to look up SAML providers only, while the latter is used
     * for OIDC providers.
     */
    type: 'saml' | 'oidc';
    /**
     * The maximum number of results to return per page. The default and maximum is
     * 100.
     */
    maxResults?: number;
    /**
     * The next page token. When not specified, the lookup starts from the beginning
     * of the list.
     */
    pageToken?: string;
}
/**
 * The request interface for updating a SAML Auth provider. This is used
 * when updating a SAML provider's configuration via
 * {@link BaseAuth.updateProviderConfig}.
 */
export interface SAMLUpdateAuthProviderRequest {
    /**
     * The SAML provider's updated display name. If not provided, the existing
     * configuration's value is not modified.
     */
    displayName?: string;
    /**
     * Whether the SAML provider is enabled or not. If not provided, the existing
     * configuration's setting is not modified.
     */
    enabled?: boolean;
    /**
     * The SAML provider's updated IdP entity ID. If not provided, the existing
     * configuration's value is not modified.
     */
    idpEntityId?: string;
    /**
     * The SAML provider's updated SSO URL. If not provided, the existing
     * configuration's value is not modified.
     */
    ssoURL?: string;
    /**
     * The SAML provider's updated list of X.509 certificated. If not provided, the
     * existing configuration list is not modified.
     */
    x509Certificates?: string[];
    /**
     * The SAML provider's updated RP entity ID. If not provided, the existing
     * configuration's value is not modified.
     */
    rpEntityId?: string;
    /**
     * The SAML provider's callback URL. If not provided, the existing
     * configuration's value is not modified.
     */
    callbackURL?: string;
}
/**
 * The request interface for updating an OIDC Auth provider. This is used
 * when updating an OIDC provider's configuration via
 * {@link BaseAuth.updateProviderConfig}.
 */
export interface OIDCUpdateAuthProviderRequest {
    /**
     * The OIDC provider's updated display name. If not provided, the existing
     * configuration's value is not modified.
     */
    displayName?: string;
    /**
     * Whether the OIDC provider is enabled or not. If not provided, the existing
     * configuration's setting is not modified.
     */
    enabled?: boolean;
    /**
     * The OIDC provider's updated client ID. If not provided, the existing
     * configuration's value is not modified.
     */
    clientId?: string;
    /**
     * The OIDC provider's updated issuer. If not provided, the existing
     * configuration's value is not modified.
     */
    issuer?: string;
    /**
     * The OIDC provider's client secret to enable OIDC code flow.
     * If not provided, the existing configuration's value is not modified.
     */
    clientSecret?: string;
    /**
     * The OIDC provider's response object for OAuth authorization flow.
     */
    responseType?: OAuthResponseType;
}
export declare type UpdateAuthProviderRequest = SAMLUpdateAuthProviderRequest | OIDCUpdateAuthProviderRequest;
/** A maximum of 10 test phone number / code pairs can be configured. */
export declare const MAXIMUM_TEST_PHONE_NUMBERS = 10;
/** The server side SAML configuration request interface. */
export interface SAMLConfigServerRequest {
    idpConfig?: {
        idpEntityId?: string;
        ssoUrl?: string;
        idpCertificates?: Array<{
            x509Certificate: string;
        }>;
        signRequest?: boolean;
    };
    spConfig?: {
        spEntityId?: string;
        callbackUri?: string;
    };
    displayName?: string;
    enabled?: boolean;
    [key: string]: any;
}
/** The server side SAML configuration response interface. */
export interface SAMLConfigServerResponse {
    name?: string;
    idpConfig?: {
        idpEntityId?: string;
        ssoUrl?: string;
        idpCertificates?: Array<{
            x509Certificate: string;
        }>;
        signRequest?: boolean;
    };
    spConfig?: {
        spEntityId?: string;
        callbackUri?: string;
    };
    displayName?: string;
    enabled?: boolean;
}
/** The server side OIDC configuration request interface. */
export interface OIDCConfigServerRequest {
    clientId?: string;
    issuer?: string;
    displayName?: string;
    enabled?: boolean;
    clientSecret?: string;
    responseType?: OAuthResponseType;
    [key: string]: any;
}
/** The server side OIDC configuration response interface. */
export interface OIDCConfigServerResponse {
    name?: string;
    clientId?: string;
    issuer?: string;
    displayName?: string;
    enabled?: boolean;
    clientSecret?: string;
    responseType?: OAuthResponseType;
}
/** The server side email configuration request interface. */
export interface EmailSignInConfigServerRequest {
    allowPasswordSignup?: boolean;
    enableEmailLinkSignin?: boolean;
}
/** Identifies the server side second factor type. */
declare type AuthFactorServerType = 'PHONE_SMS';
/** Server side multi-factor configuration. */
export interface MultiFactorAuthServerConfig {
    state?: MultiFactorConfigState;
    enabledProviders?: AuthFactorServerType[];
}
/**
 * Identifies a second factor type.
 */
export declare type AuthFactorType = 'phone';
/**
 * Identifies a multi-factor configuration state.
 */
export declare type MultiFactorConfigState = 'ENABLED' | 'DISABLED';
/**
 * Interface representing a multi-factor configuration.
 * This can be used to define whether multi-factor authentication is enabled
 * or disabled and the list of second factor challenges that are supported.
 */
export interface MultiFactorConfig {
    /**
     * The multi-factor config state.
     */
    state: MultiFactorConfigState;
    /**
     * The list of identifiers for enabled second factors.
     * Currently only ‘phone’ is supported.
     */
    factorIds?: AuthFactorType[];
}
/**
 * Defines the multi-factor config class used to convert client side MultiFactorConfig
 * to a format that is understood by the Auth server.
 */
export declare class MultiFactorAuthConfig implements MultiFactorConfig {
    readonly state: MultiFactorConfigState;
    readonly factorIds: AuthFactorType[];
    /**
     * Validates the MultiFactorConfig options object. Throws an error on failure.
     *
     * @param options - The options object to validate.
     */
    private static validate;
    /** @returns The plain object representation of the multi-factor config instance. */
    toJSON(): object;
}
/**
 * Validates the provided map of test phone number / code pairs.
 * @param testPhoneNumbers - The phone number / code pairs to validate.
 */
export declare function validateTestPhoneNumbers(testPhoneNumbers: {
    [phoneNumber: string]: string;
}): void;
/**
 * The email sign in provider configuration.
 */
export interface EmailSignInProviderConfig {
    /**
     * Whether email provider is enabled.
     */
    enabled: boolean;
    /**
     * Whether password is required for email sign-in. When not required,
     * email sign-in can be performed with password or via email link sign-in.
     */
    passwordRequired?: boolean;
}
/**
 * The base Auth provider configuration interface.
 */
export interface BaseAuthProviderConfig {
    /**
     * The provider ID defined by the developer.
     * For a SAML provider, this is always prefixed by `saml.`.
     * For an OIDC provider, this is always prefixed by `oidc.`.
     */
    providerId: string;
    /**
     * The user-friendly display name to the current configuration. This name is
     * also used as the provider label in the Cloud Console.
     */
    displayName?: string;
    /**
     * Whether the provider configuration is enabled or disabled. A user
     * cannot sign in using a disabled provider.
     */
    enabled: boolean;
}
/**
 * The
 * [SAML](http://docs.oasis-open.org/security/saml/Post2.0/sstc-saml-tech-overview-2.0.html)
 * Auth provider configuration interface. A SAML provider can be created via
 * {@link BaseAuth.createProviderConfig}.
 */
export interface SAMLAuthProviderConfig extends BaseAuthProviderConfig {
    /**
     * The SAML IdP entity identifier.
     */
    idpEntityId: string;
    /**
     * The SAML IdP SSO URL. This must be a valid URL.
     */
    ssoURL: string;
    /**
     * The list of SAML IdP X.509 certificates issued by CA for this provider.
     * Multiple certificates are accepted to prevent outages during
     * IdP key rotation (for example ADFS rotates every 10 days). When the Auth
     * server receives a SAML response, it will match the SAML response with the
     * certificate on record. Otherwise the response is rejected.
     * Developers are expected to manage the certificate updates as keys are
     * rotated.
     */
    x509Certificates: string[];
    /**
     * The SAML relying party (service provider) entity ID.
     * This is defined by the developer but needs to be provided to the SAML IdP.
     */
    rpEntityId: string;
    /**
     * This is fixed and must always be the same as the OAuth redirect URL
     * provisioned by Firebase Auth,
     * `https://project-id.firebaseapp.com/__/auth/handler` unless a custom
     * `authDomain` is used.
     * The callback URL should also be provided to the SAML IdP during
     * configuration.
     */
    callbackURL?: string;
}
/**
 * The interface representing OIDC provider's response object for OAuth
 * authorization flow.
 * One of the following settings is required:
 * <ul>
 * <li>Set <code>code</code> to <code>true</code> for the code flow.</li>
 * <li>Set <code>idToken</code> to <code>true</code> for the ID token flow.</li>
 * </ul>
 */
export interface OAuthResponseType {
    /**
     * Whether ID token is returned from IdP's authorization endpoint.
     */
    idToken?: boolean;
    /**
     * Whether authorization code is returned from IdP's authorization endpoint.
     */
    code?: boolean;
}
/**
 * The [OIDC](https://openid.net/specs/openid-connect-core-1_0-final.html) Auth
 * provider configuration interface. An OIDC provider can be created via
 * {@link BaseAuth.createProviderConfig}.
 */
export interface OIDCAuthProviderConfig extends BaseAuthProviderConfig {
    /**
     * This is the required client ID used to confirm the audience of an OIDC
     * provider's
     * [ID token](https://openid.net/specs/openid-connect-core-1_0-final.html#IDToken).
     */
    clientId: string;
    /**
     * This is the required provider issuer used to match the provider issuer of
     * the ID token and to determine the corresponding OIDC discovery document, eg.
     * [`/.well-known/openid-configuration`](https://openid.net/specs/openid-connect-discovery-1_0.html#ProviderConfig).
     * This is needed for the following:
     * <ul>
     * <li>To verify the provided issuer.</li>
     * <li>Determine the authentication/authorization endpoint during the OAuth
     *     `id_token` authentication flow.</li>
     * <li>To retrieve the public signing keys via `jwks_uri` to verify the OIDC
     *     provider's ID token's signature.</li>
     * <li>To determine the claims_supported to construct the user attributes to be
     *     returned in the additional user info response.</li>
     * </ul>
     * ID token validation will be performed as defined in the
     * [spec](https://openid.net/specs/openid-connect-core-1_0.html#IDTokenValidation).
     */
    issuer: string;
    /**
     * The OIDC provider's client secret to enable OIDC code flow.
     */
    clientSecret?: string;
    /**
     * The OIDC provider's response object for OAuth authorization flow.
     */
    responseType?: OAuthResponseType;
}
/**
 * The Auth provider configuration type.
 * {@link BaseAuth.createProviderConfig}.
 */
export declare type AuthProviderConfig = SAMLAuthProviderConfig | OIDCAuthProviderConfig;
export {};
